<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Statement activity object for xAPI structure checking and usage.
 *
 * @package    core_xapi
 * @copyright  2020 Ferran Recio
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace core_xapi\local\statement;

use core_xapi\xapi_exception;
use core_xapi\iri;
use stdClass;

defined('MOODLE_INTERNAL') || die();

/**
 * Class that implements a xAPI activity compatible with xAPI object.
 *
 * @copyright  2020 Ferran Recio
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class item_activity extends item_object {

    /** @var string Activity ID. */
    protected $id;

    /** @var item_definition Definition object. */
    protected $definition;

    /**
     * Item activity constructor.
     *
     * An xAPI activity is mainly an IRI ID and an optional definition.
     *
     * @param stdClass $data from the specific xAPI element
     * @param item_definition $definition option definition item
     */
    protected function __construct(stdClass $data, ?item_definition $definition = null) {
        parent::__construct($data);
        $this->id = iri::extract($data->id, 'activity');
        $this->definition = $definition;
    }

    /**
     * Function to create an item from part of the xAPI statement.
     *
     * @param stdClass $data the original xAPI element
     * @return item item_activity xAPI generated
     */
    public static function create_from_data(stdClass $data): item {
        if (!isset($data->objectType)) {
            throw new xapi_exception('Missing activity objectType');
        }
        if ($data->objectType != 'Activity') {
            throw new xapi_exception('Activity objectType must be "Activity"');
        }
        if (empty($data->id)) {
            throw new xapi_exception("Missing Activity id");
        }
        if (!iri::check($data->id)) {
            throw new xapi_exception("Activity id $data->id is not a valid IRI");
        }

        $definition = null;
        if (!empty($data->definition)) {
            $definition = item_definition::create_from_data($data->definition);
        }

        return new self($data, $definition);
    }

    /**
     * Generate a valid item_activity from a simple ID string and an optional definition.
     *
     * @param string $id any string that will converted into a valid IRI
     * @param item_definition|null $definition optional item_definition
     * @return item_activity
     */
    public static function create_from_id(string $id, ?item_definition $definition = null): item_activity {
        $data = (object) [
            'objectType' => 'Activity',
            'id' => iri::generate($id, 'activity'),
        ];

        if (!empty($definition)) {
            $data->definition = $definition->get_data();
        }

        return new self($data, $definition);
    }

    /**
     * Return the activity ID.
     *
     * If the ID was generated by iri::generate this function will return
     * the iri:extract value.
     *
     * @return string the activity ID
     */
    public function get_id(): string {
        return $this->id;
    }

    /**
     * Returns the item_definition of this item.
     *
     * @return item_definition|null the item definition if available
     */
    public function get_definition(): ?item_definition {
        return $this->definition;
    }
}
